<?php

namespace MathPHP\Tests\Statistics\Multivariate\PLS;

use MathPHP\LinearAlgebra\Matrix;
use MathPHP\LinearAlgebra\MatrixFactory;
use MathPHP\SampleData;
use MathPHP\Statistics\Multivariate\PLS;

class CerealScaleFalseTest extends \PHPUnit\Framework\TestCase
{
    /** @var PLS */
    private static $pls;

    /** @var Matrix */
    private static $X;

    /** @var Matrix */
    private static $Y;

    /**
     * R code for expected values:
     *   library(chemometrics)
     *   data(cereal)
     *   pls.model = pls2_nipals(cereal$X, cereal$Y, a=5)
     *
     * @throws Exception\MathException
     */
    public static function setUpBeforeClass(): void
    {
        $cereal   = new SampleData\Cereal();
        self::$X  = MatrixFactory::create($cereal->getXData());
        self::$Y  = MatrixFactory::create($cereal->getYData());

        self::$pls = new PLS(self::$X, self::$Y, 5, false);
    }

    /**
     * @test         Construction
     * @throws       Exception\MathException
     */
    public function testConstruction()
    {
        // When
        $pls = new PLS(self::$X, self::$Y, 5, false);

        // Then
        $this->assertInstanceOf(PLS::class, $pls);
    }

    /**
     * @test The class returns the correct values for T
     *
     * R code for expected values:
     *   pls.model$T
     */
    public function testT()
    {
        // Given
        $expected = [
            [-0.0001868638, -0.0002046034, 0.0046609248, -0.0001600709, 0.00004982759],
            [-0.0016778554, 0.00005917951, 0.0052769064, -0.0006910983, 0.0004655183],
            [-0.0051548113, -0.000447606, 0.0011327825, 0.0004474364, -0.000007108459],
            [0.0145545101, 0.0005525388, -0.0042893817, 0.0004234981, 0.0009071037],
            [0.0130044547, 0.001629171, -0.0007747091, -0.001552795, 0.0008151882],
            [0.0190486865, 0.002853815, -0.0011286369, -0.0008433846, -0.0004428237],
            [-0.0241767255, -0.001024907, -0.0023870719, -0.000895506, 0.0009098762],
            [-0.0115948395, -0.0003388141, -0.0008535249, -0.001196682, -0.001003641],
            [-0.0227801577, -0.0008897606, -0.0033519487, 0.00004398769, 0.0004707373],
            [0.0116569657, -0.002063344, 0.0004043838, -0.001575964, -0.000555956],
            [0.020761095, -0.00165762, 0.0008174193, 0.00006268997, -0.001051075],
            [-0.0127461466, 0.0006911177, -0.0030157056, 0.0005838427, -0.0004274284],
            [-0.0222939597, 0.001663128, 0.0050029883, 0.00116166, 0.0000582372],
            [-0.00147144, 0.0004688227, -0.002098463, 0.002020786, -0.001425581],
            [0.0230570875, -0.001291118, 0.0006040369, 0.002171599, 0.001237125],
        ];

        // When
        $T = self::$pls->getXScores()->getMatrix();

        // Then
        $this->assertEqualsWithDelta($expected, $T, .00001, '');
    }

    /**
     *  @test The class returns the correct values for W
     *
     * R code for expected values:
     *   pls.model$W
     */
    public function testW()
    {
        // Given
        $expected = [
            [0.0734291936, 0.075494873, -0.07962752, -0.0991457361, 0.0185171419],
            [0.0892686597, 0.084473465, -0.1127286, -0.1365435312, -0.0176498597],
            [0.0974581612, 0.088332581, -0.1287691, -0.1560144902, -0.0487064718],
            [0.0982980659, 0.088679592, -0.1257427, -0.1471835056, -0.0566276198],
            [0.0932582064, 0.08755351, -0.1125333, -0.1276608447, -0.062074792],
            [0.083935107, 0.084074163, -0.09434272, -0.1022785269, -0.0531962013],
            [0.0717861972, 0.080785526, -0.0754114, -0.0752361679, -0.0317767519],
            [0.0564547084, 0.076293373, -0.05392677, -0.0486804428, -0.0003315844],
            [0.0342801754, 0.055114963, -0.03009046, -0.0283350103, 0.0188528805],
            [0.0051755787, 0.011374032, -0.004623049, -0.0241674056, -0.0008129216],
            [-0.0249559994, -0.044928302, 0.02023768, -0.0207772221, -0.0291625869],
            [-0.047249076, -0.091188641, 0.03862492, -0.0195085463, -0.0591948108],
            [-0.0570183961, -0.114887066, 0.04721758, -0.0206189599, -0.0739206142],
            [-0.0540608616, -0.110269236, 0.04812857, -0.018392109, -0.0606979815],
            [-0.0423090386, -0.084893256, 0.04428089, -0.0123256, -0.0283547526],
            [-0.0278503015, -0.05212947, 0.03857857, -0.0043326936, 0.0092310907],
            [-0.0164422686, -0.02682368, 0.03308704, -0.0022953127, 0.0268540255],
            [-0.0099516813, -0.013655314, 0.02949484, -0.0014351622, 0.0328966545],
            [-0.0072966086, -0.011876062, 0.02760579, -0.0017786045, 0.0318196161],
            [-0.00489996, -0.01233647, 0.02670553, -0.0014160963, 0.0363670274],
            [-0.0005196826, -0.010317084, 0.02318916, -0.0057241907, 0.0400761918],
            [0.0063721714, -0.008349051, 0.01511423, -0.0165365073, 0.0328595551],
            [0.016709234, -0.004474031, 0.002936339, -0.0337741804, 0.0200435275],
            [0.0307736152, 0.003512388, -0.01292375, -0.0553133615, 0.0010493804],
            [0.0472873321, 0.016687569, -0.03019209, -0.0779703691, -0.0260145151],
            [0.064046451, 0.03370313, -0.04276724, -0.0903051936, -0.0467713595],
            [0.079186419, 0.054041371, -0.04717635, -0.0874471834, -0.0528755357],
            [0.0908738628, 0.074759744, -0.04634188, -0.0771151977, -0.0509725576],
            [0.0988677214, 0.093343685, -0.0421365, -0.064729227, -0.0395171013],
            [0.1054065648, 0.10668808, -0.03450237, -0.0542116389, -0.0215019345],
            [0.1157522518, 0.113391641, -0.02697749, -0.0558662343, -0.0059630096],
            [0.1338489794, 0.110315553, -0.02362988, -0.074862807, -0.0032697251],
            [0.1614583209, 0.102458087, -0.02661343, -0.0994672395, -0.013072058],
            [0.1945841618, 0.095230225, -0.03543942, -0.1122939437, -0.040613067],
            [0.222921213, 0.085020556, -0.05230637, -0.1091239881, -0.0843599019],
            [0.2366489948, 0.073667929, -0.07328986, -0.0822778341, -0.1033759532],
            [0.2282272597, 0.056093051, -0.09002989, -0.0364351076, -0.0977300882],
            [0.1970405588, 0.027039937, -0.09593007, 0.000133856, -0.0967415997],
            [0.1516202077, -0.00688854, -0.08697914, 0.017341027, -0.0899662683],
            [0.103744451, -0.036631193, -0.06574509, 0.0184011668, -0.0642627691],
            [0.0628008611, -0.050988831, -0.03650629, 0.0132840752, -0.0307380903],
            [0.0327153902, -0.046960997, -0.006122613, 0.0073512444, -0.0046087531],
            [0.0137150705, -0.024400173, 0.02165262, 0.0159392009, 0.0346546338],
            [0.001604044, 0.004564669, 0.04108197, 0.0330224688, 0.0761993905],
            [-0.0084975252, 0.02286459, 0.05107148, 0.0443441052, 0.0946472688],
            [-0.0170939888, 0.030149721, 0.05187104, 0.0443056428, 0.0889575765],
            [-0.0243135898, 0.024842092, 0.04191145, 0.0316200478, 0.0701487974],
            [-0.0297346711, 0.010908754, 0.0244106, 0.0107799084, 0.0532020772],
            [-0.032322075, -0.003797749, 0.003814019, -0.0138719995, 0.0402027741],
            [-0.0322299704, -0.015534321, -0.0154572, -0.0319995338, 0.0311446733],
            [-0.031022367, -0.025843393, -0.03146223, -0.0471712452, 0.0181355733],
            [-0.0285154046, -0.031649904, -0.03982021, -0.054133258, 0.0094616403],
            [-0.0251017169, -0.031493259, -0.03981183, -0.0498236953, 0.0111996703],
            [-0.0224073882, -0.027601519, -0.03425867, -0.041755306, 0.0154633488],
            [-0.0218947549, -0.024085434, -0.02559319, -0.0380336611, 0.0163796822],
            [-0.0241590379, -0.020674365, -0.01467368, -0.0367333863, 0.0169315971],
            [-0.0293929459, -0.018831412, -0.004121736, -0.0369800284, 0.0116822293],
            [-0.0373465053, -0.022596546, 0.003958847, -0.0415607069, -0.0011909358],
            [-0.046047902, -0.028606299, 0.01094916, -0.0489955001, -0.0237719969],
            [-0.0534470496, -0.034222651, 0.01891616, -0.0465878619, -0.038390212],
            [-0.0583779795, -0.03904415, 0.02589318, -0.040057806, -0.0439165814],
            [-0.0597371569, -0.040162385, 0.03161584, -0.0352263691, -0.0427921247],
            [-0.0571977721, -0.035628136, 0.03700254, -0.0296002018, -0.0314716758],
            [-0.050721145, -0.024416561, 0.04424005, -0.0171318853, -0.005454985],
            [-0.0402625978, -0.0040815, 0.05498803, 0.0047933988, 0.0320427902],
            [-0.0255901207, 0.026700872, 0.06897269, 0.0325997951, 0.0627696397],
            [-0.0069535852, 0.060022672, 0.08325015, 0.0674861173, 0.0750819371],
            [0.0157301459, 0.090357444, 0.0893479, 0.097856044, 0.0701984984],
            [0.0410114259, 0.122056876, 0.0813847, 0.1172455269, 0.0586074403],
            [0.0639032885, 0.154599165, 0.06127667, 0.1191099308, 0.0524359187],
            [0.0773158296, 0.175004723, 0.04071865, 0.1103008184, 0.0571533319],
            [0.077895887, 0.17401285, 0.03164864, 0.1013778551, 0.0603112644],
            [0.064957818, 0.14318625, 0.0306274, 0.0823765702, 0.0346626979],
            [0.0405774539, 0.072742113, 0.02919583, 0.0467017878, -0.0373616251],
            [0.0162101859, -0.011428077, 0.01884371, -0.0023472591, -0.1198390536],
            [0.0012814899, -0.076072065, -0.002281165, -0.0491685549, -0.169357725],
            [-0.0033845096, -0.10671719, -0.02895817, -0.0889423107, -0.1697025274],
            [-0.0025551693, -0.10821809, -0.04854639, -0.1054989175, -0.1129504691],
            [-0.0002382727, -0.091882314, -0.05177462, -0.0998399114, -0.043219179],
            [-0.0010565975, -0.070834481, -0.03669841, -0.0763788785, 0.0119713826],
            [-0.0095028058, -0.065918716, -0.0111472, -0.0529704812, 0.0189332996],
            [-0.021756916, -0.070379542, 0.01325369, -0.0437026753, -0.0166032006],
            [-0.0308851017, -0.062419732, 0.03569473, -0.0292545257, -0.0315144345],
            [-0.0355555603, -0.048031982, 0.05206415, -0.0139807927, -0.0215662698],
            [-0.0366949944, -0.034235647, 0.06075533, -0.0005466546, 0.0092145626],
            [-0.0358095414, -0.026902939, 0.059677, 0.0006451073, 0.0314035028],
            [-0.0348045927, -0.027966295, 0.05327137, -0.0030940643, 0.044300105],
            [-0.033786543, -0.03560796, 0.0458226, -0.0102255803, 0.0454047791],
            [-0.0301252153, -0.044259428, 0.03888341, -0.0278714572, 0.0196667351],
            [-0.0190014439, -0.044679641, 0.03717849, -0.036805201, 0.008521681],
            [0.0028983721, -0.045752862, 0.03730737, -0.0400997437, 0.0080578938],
            [0.038766527, -0.049802677, 0.03992511, -0.0471164703, 0.0164330432],
            [0.0913798138, -0.050543209, 0.05396866, -0.0428052043, 0.0425905771],
            [0.1554253141, -0.062748273, 0.08124805, -0.0195276845, 0.0588607383],
            [0.2184605996, -0.095763518, 0.1109858, 0.0203293535, 0.042049592],
            [0.2632006065, -0.147028855, 0.1318394, 0.0732552457, 0.002332687],
            [0.2764944403, -0.205172829, 0.134274, 0.1286126516, -0.0368880188],
            [0.2548888592, -0.254549044, 0.1084618, 0.1594024565, -0.0792129907],
            [0.2034643754, -0.274495, 0.06503745, 0.1549841857, -0.1095217052],
            [0.1356474004, -0.248811705, 0.02402457, 0.1144503991, -0.1211200999],
            [0.0692310929, -0.181034006, 0.00134473, 0.0554733352, -0.1086416493],
            [0.0162257885, -0.091921914, 0.00006780712, 0.0006236463, -0.1109838],
            [-0.0224118069, -0.003745113, 0.01794114, -0.0279055435, -0.1279779774],
            [-0.049801375, 0.064912722, 0.04531234, -0.0302688845, -0.1583378314],
            [-0.0670249943, 0.104752611, 0.0616016, -0.0195391763, -0.1721634296],
            [-0.0761482961, 0.120521813, 0.05807035, -0.0080644355, -0.1617751481],
            [-0.0798577889, 0.117000704, 0.03774134, -0.0044924284, -0.1394618248],
            [-0.078046641, 0.104065211, 0.003210515, -0.0140995852, -0.1059133758],
            [-0.0707440263, 0.08610324, -0.03311872, -0.0078222522, -0.0600676801],
            [-0.059410679, 0.064080389, -0.06511419, 0.0109529424, -0.0556557039],
            [-0.0447731358, 0.042648947, -0.08337773, 0.0446358886, -0.1049610537],
            [-0.0268986934, 0.026384721, -0.08461547, 0.1013526043, -0.1523031555],
            [-0.008829279, 0.013944395, -0.07998647, 0.1480567483, -0.1646065668],
            [0.0054035693, -0.004375208, -0.08053276, 0.1708822429, -0.1351551377],
            [0.0156041629, -0.022382477, -0.08746085, 0.1684728348, -0.068016786],
            [0.0211956112, -0.036225777, -0.0997581, 0.1497838509, 0.0443422028],
            [0.0207553085, -0.051821554, -0.1162756, 0.1129728707, 0.1427496775],
            [0.0155084915, -0.067640179, -0.1295653, 0.0604543702, 0.1695933063],
            [0.0080940162, -0.077330038, -0.1301662, 0.0058635689, 0.1358249266],
            [0.0025906578, -0.063366986, -0.1082072, -0.0221938019, 0.1242855411],
            [-0.0026157959, -0.035770572, -0.07226682, -0.0232218162, 0.1266829189],
            [-0.0077551526, -0.0018186, -0.02654082, -0.0051787071, 0.1328430886],
            [-0.0126880265, 0.037794667, 0.02273443, 0.0237221654, 0.1241062199],
            [-0.0192791832, 0.075716831, 0.06565285, 0.0523065803, 0.0757348869],
            [-0.0267081425, 0.108402668, 0.101632, 0.0809234923, -0.0065824588],
            [-0.0347290346, 0.129472817, 0.1325252, 0.1074486763, -0.0991600375],
            [-0.0467326376, 0.122139423, 0.1412744, 0.1162661463, -0.1750596724],
            [-0.0607264466, 0.096862205, 0.1274396, 0.1132370262, -0.2258964506],
            [-0.0746823613, 0.057360542, 0.0920283, 0.089426683, -0.254001904],
            [-0.0857967372, 0.01721135, 0.04905672, 0.0553036479, -0.2449260088],
            [-0.0944404223, -0.019332691, -0.0009352124, 0.0254404436, -0.2270182304],
            [-0.0976040161, -0.044217916, -0.04538688, 0.0112302568, -0.1517925164],
            [-0.0930266811, -0.066162405, -0.07325798, 0.0127330806, -0.0681817118],
            [-0.0826153817, -0.093524564, -0.1042782, 0.0083553534, -0.0184803979],
            [-0.0613757269, -0.093811693, -0.1358363, 0.0255647638, 0.0281159389],
            [-0.0304501518, -0.059972048, -0.1602326, 0.0816008188, 0.0540665159],
            [-0.0003629197, -0.001148256, -0.1927557, 0.1339126004, 0.0364610867],
            [0.0170313272, 0.043833289, -0.2314117, 0.1846037508, -0.0215888405],
            [0.0205111872, 0.081434444, -0.2380915, 0.267988531, -0.0081098885],
            [0.0073955253, 0.075359084, -0.2367175, 0.2940863182, -0.0090193797],
            [-0.0204329354, 0.015333511, -0.2269116, 0.2341934231, -0.0307931174],
            [-0.0515831837, -0.044355505, -0.2136522, 0.1286993512, 0.0080038775],
            [-0.0830750777, -0.100177206, -0.1924895, 0.0035790367, -0.0226403313],
            [-0.1078793779, -0.136461889, -0.1961337, -0.1191571872, -0.0387444115],
            [-0.1209873211, -0.158447572, -0.2098556, -0.2311886647, -0.0726185357],
        ];

        // When
        $W = self::$pls->getXLoadings()->getMatrix();

        // Then
        $this->assertEqualsWithDelta($expected, $W, .00001, '');
    }
}
